/**
* \file: AditVideoSink.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* Video Sink Endpoint
*
* \component: Baidu CarLife
*
* \author: P. Govindaraju / RBEB/GM / Pradeepa.Govindaraju@in.bosch.com
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef BDCL_ADITVIDEOSINK_H_
#define BDCL_ADITVIDEOSINK_H_

#include <string>
#include "bdclErrorCodes.h"

namespace adit { namespace bdcl {

class CoreCallbackDealer;

/** @interface  IAditVideoSinkCallbacks
 *  @brief      Callbacks to notify the MC application. Most of the callbacks are initiated by Baidu stack and match
 *              their signature. This interface must be implemented by the MC application using ADIT Video Sink
 *              implementation.
 */
class IAditVideoSinkCallbacks
{
public:
    virtual ~IAditVideoSinkCallbacks() { }

    /**
     * @brief Callback initiated by Video Endpoint to inform MC about rendering of
     * the first frame.
     */
    virtual void onFirstFrameRendered() { };

    /** @brief Callback initiated by Baidu core to acknowledge the VideoEnocderInit call. The input parameters must be
     *         used to configure the Gstreamer pipline if initially configured parameters are denied by MD.
     */
    virtual void onVideoEncoderInitDone(unsigned int inWidth, unsigned int inHeight, unsigned int inFramerate)
    {
        (void)inHeight;
        (void)inWidth;
        (void)inFramerate;
    };

    /** @brief Callback initiated by Baidu core to after calling VideoEnocderInit, if MD not faces soome
     * problem to start the video decoder.
     */
    virtual void onConnectException(carlifeConnectException inConnectException) { (void) inConnectException; };

    /** @brief Callback initiated by ADIT to inform MC application about an error.
     */
    virtual void onError(bdclErrorCodes inErrorCode) { (void) inErrorCode; };

};

/** @class      AditVideoSink
 *  @brief      Abstract video sink interface to be used by the MC. MC has to implement a class of a concrete
 *              implementation of AditVideoSink, but use AditVideoSink interfaces for easier portability.
 */
class AditVideoSink
{
public:
    // after the teardown is called, it has to be ensured that these callbacks won't be triggered anymore.
    // So MC is free to destroy them, but MC has to guarantee that the IAditVideoSinkCallbacks object is alive before
    // they call the teardown.

    /** @brief AditVideoSink constructor
     *  @param inCallbacks: object implementing the callback interface. MC has to ensure that IAditVideoSinkCallbacks
     *         object is alive until they call teardown.
     */
    AditVideoSink(IAditVideoSinkCallbacks* inCallbacks, CoreCallbackDealer* inCallbackDealer)
    {
        (void)inCallbacks;
        (void)inCallbackDealer;
    }

    virtual ~AditVideoSink() { };

    //the config params are:
    //      int threadPriority, bool DisableThreadPriority,
    /** @brief Configuration interface for each parameter
     *  @param inKey: name of the configuration parameter
     *       - video-threads-real-time-priority [optional]: Thread priority for video rendering thread.
     *       - disable-real-time-priority-video [optional]: setting this flag means video rendering thread takes default thread priority.
     *       - video-width [mandatory]: The number of pixels in the x axis for a logical UI resolution.
     *       - video-height [mandatory]: The number of pixels in the y axis for a logical UI resolution.
     *       - video-fps [mandatory]: Frame rate of video. (must be "30" or "60").
     *       - gstreamer-video-pipeline [mandatory]: complete pipleline EX:"vpudec low-latency=true frame-plus=2 framedrop=false framerate-nu=30 dis-reorder=true ! "
            "gst_apx_sink display-width=800 display-height=480 sync=false qos=false max-lateness=3000000000"
     *  @param inValue: the actual value for configuration
     */
    virtual void setConfigItem(std::string inKey, std::string inValue) = 0;

    /** @brief Method to initialize video sink endpoint. Inside this function Baidu stack API cmdVideoEncoderInit()
     * will be called. MC has to ensure that the configuration is done before calling initialize.
     *  @return true on success
     *          false on failure
     */
    virtual bool initialize() = 0;
    /** @brief Method to shutdown video sink endpoint. After teardown call, MC is free to destroy
     *         IAditVideoSinkCallbacks object
     */
    virtual void teardown() = 0;

    /** @brief Method to inform the MD to start the Encoder on agreed Height, Width and Framerate.
     * MC has to ensure that the reconfiguration of pipeline is done before calling video encoder start. This reconfiguration
     * has to be done only if MD returns with different configuration values through initVideoEncoderDone().
     */
    virtual bool playbackStart() = 0;
    /** @brief Method to stop video streaming from MD.
     */
    virtual void playbackPause() = 0;
    /** @brief Method to call cmdVideoEncoderInit(). MC has to call this after initializing of core has returned
     *         successfully.
    */
    virtual void configureStream() = 0;
};

} } /* namespace adit { namespace bdcl { */




#endif /* BDCL_ADITVIDEOSINK_H_ */
